/*
 * cSimMsgHeap.cpp
 * 
 * Class implementation for a message delivery heap.
 *
 */
#include "cSimMsgHeap.h"

#ifndef NULL
	#define NULL 0
#endif

/*
 * cSimMsgHeap::cSimMsgHeap()
 *
 * Purpose:	Creates a new heap.
 * IN:		initialCap	-> The starting size of the queue.
 *			capInc		-> How much to grow the queue by every time.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is set up.
 * Throws:	-
 * Return:	-
 */
cSimMsgHeap::cSimMsgHeap(unsigned int initialCap, unsigned int capInc)
{
	mSize			= initialCap;
	mHeapSize		= 0;
	mCapIncrement   = capInc;

	// Alloc queue space.
	mQueue = new SimMsg[mSize];
	if(!mQueue)
	{
		//throw cQueueException("<cSortedMsgQueue::cSortedMsgQueue>: Unable to alloc new queue.");
	}
//	for(unsigned int i = 0; i < mSize; i++)
//	{
//		mQueue[i].msg = NULL;
//		mQueue[i].destEp = NULL;
//	}
}

/*
 * cSimMsgHeap::~cSimMsgHeap()
 *
 * Purpose:	Destroys the heap.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is cleaned up.
 * Return:	-
 */
cSimMsgHeap::~cSimMsgHeap()
{
	if(mQueue) 
	{ 
		Clear();
		delete mQueue; 
	}
}

/*
 * cSimMsgHeap::_Grow()
 *
 * Purpose:	Attempts to grow the queue by at least one, at most mCapIncrement
 * IN:		-
 * OUT:		-
 * Cond:	The queue is full!
 * PostCnd:	The queue is grown by at least one, at most mCapIncrement elements.
 * Return:	true if success, else false.
 */ 
bool cSimMsgHeap::_Grow()
{
	// We need to expand the queue.
	SimMsg* newQ = new SimMsg[mSize+mCapIncrement];
	if(!newQ)
	{
		return false;
	}

	// The following may be faster using memcpy()...
	for(unsigned int i = 0; i < mSize; i++)
	{
		newQ[i] = mQueue[i];
	}
	mSize = mSize + mCapIncrement;
	delete mQueue;
	mQueue = newQ;

 	return true;
}

/*
 * cSimMsgHeap::Insert()
 *
 * Purpose:	Attempts to insert the node, but only if it is not in the view.
 * IN:		node	-> The node to insert.
 * OUT:		-
 * Cond:	The node is not in the queue.
 * PostCnd:	The node is placed in the queue.
 * Return:	true if success, else false.
 */ 
bool cSimMsgHeap::Insert(SimMsg* msg)
{
	unsigned int i;
	mHeapSize++;
	if(mHeapSize == mSize)
	{
		if(!_Grow())
		{
			return false;
		}
	}
	i = mHeapSize-1;
	while( (i > 0) && (mQueue[_Parent(i)].timeToDeliver > msg->timeToDeliver) )
	{
		mQueue[i] = mQueue[_Parent(i)];	// Switch parent and new entry.
		i = _Parent(i);
	}
	mQueue[i] = *msg;
	return true;
}

/*
 * cSimMsgHeap::Clear()
 *
 * Purpose:	Clears out the heap.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The heap is reset and cleared.
 * Return:	true if success, false otherwise.
 */
bool cSimMsgHeap::Clear()
{
	for(unsigned int i = 0; i < mSize; i++)
	{
		if(mQueue[i].destEp)
		{
//			mQueue[i].destEp = NULL;
//			mQueue[i].msg = NULL;
//			mQueue[i].sendEp = NULL;
		}
	}
	mHeapSize = 0;
	return true;
}

/*
 * cSimMsgHeap::Remove()
 *
 * Purpose:	Removes the element with the smallest time to deliver.
 * IN:		-
 * OUT:		msg		-> A copy of the SimMsg
 * Cond:	There is an element in the queue.
 * PostCnd:	The queue is shrunk by one element.
 * Return:	true if success, else false.
 */ 
bool cSimMsgHeap::Remove(SimMsg* msg)
{
	if(mHeapSize)
	{
		*msg = mQueue[0];
		mQueue[0] = mQueue[mHeapSize-1];
//		mQueue[mHeapSize-1].destEp = NULL;
//		mQueue[mHeapSize-1].msg    = NULL;
//		mQueue[mHeapSize-1].sendEp = NULL;
		mHeapSize--;
		_Heapify(0);
		return true;
	}
	return false;
}

/*
 * cSimMsgHeap::Peek()
 *
 * Purpose:	Looks at the element with the smallest time to deliver.
 * IN:		-
 * OUT:		msg		-> A copy of the SimMsg
 * Cond:	There is an element in the queue.
 * PostCnd:	-
 * Return:	true if success, else false.
 */ 
bool cSimMsgHeap::Peek(SimMsg* msg)
{
	if(mHeapSize)
	{
		*msg = mQueue[0];
		return true;
	}
	return false;
}

/*
 * cSimMsgHeap::_Exchange()
 *
 * Purpose:	Exchanges two elements in the heap.
 * IN:		a, b		-> The indices of the elements to exchange.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success, else false.
 */ 
bool cSimMsgHeap::_Exchange(unsigned int a, unsigned int b)
{
	SimMsg	temp;
	temp = mQueue[a];
	mQueue[a] = mQueue[b];
	mQueue[b] = temp;
	return true;
}

/*
 * cSimMsgHeap::_Heapify()
 *
 * Purpose:	Recreates the heap property.
 * IN:		index		-> The index of the root of the tree to heapify.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success, else false.
 */ 
bool cSimMsgHeap::_Heapify(unsigned int index)
{
	unsigned int left, right;
	unsigned int chosen = index;

	while(1)
	{
		left  = _Left(index);
		right = _Right(index);
		if((left < mHeapSize) && (mQueue[left].timeToDeliver < mQueue[index].timeToDeliver))
		{
			chosen = left;
		}
		if((right < mHeapSize) && (mQueue[right].timeToDeliver < mQueue[chosen].timeToDeliver))
		{
			chosen = right;
		}
		if(chosen != index)
		{
			_Exchange(chosen, index);
			index = chosen;		// Start from the chosen one now.
		}
		else
		{
			return true;	// done
		}
	} 
	return false;
}